package gov.va.med.mhv.usermgmt.service.impl;

import gov.va.med.mhv.core.service.delegate.EmailServiceDelegate;
import gov.va.med.mhv.core.util.Precondition;
import gov.va.med.mhv.core.util.TimestampUtils;
import gov.va.med.mhv.service.MHVAbstractService;
import gov.va.med.mhv.usermgmt.bizobj.BusinessObjectFactory;
import gov.va.med.mhv.usermgmt.bizobj.EAuthHashBO;
import gov.va.med.mhv.usermgmt.bizobj.EAuthHashBaseBO;
import gov.va.med.mhv.usermgmt.bizobj.TermsConditionsVerHistBO;
import gov.va.med.mhv.usermgmt.bizobj.UserAcceptTermsHistoryBO;
import gov.va.med.mhv.usermgmt.bizobj.UserPasswordHistoryBO;
import gov.va.med.mhv.usermgmt.bizobj.UserProfileAssembler;
import gov.va.med.mhv.usermgmt.bizobj.UserProfileBO;
import gov.va.med.mhv.usermgmt.bizobj.UserProfileDeactivationAuditAssembler;
import gov.va.med.mhv.usermgmt.bizobj.UserProfileDeactivationAuditBO;
import gov.va.med.mhv.usermgmt.bizobj.UserProfileDeactivationReasonAssembler;
import gov.va.med.mhv.usermgmt.bizobj.UserProfileDeactivationReasonBO;
import gov.va.med.mhv.usermgmt.enumeration.ActivityActionTypeEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.ActivityActorTypeEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.ActivityTypeEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.GenderEnumeration;
import gov.va.med.mhv.usermgmt.messages.UserManagementMessages;
import gov.va.med.mhv.usermgmt.service.EAuthHashServiceResponse;
import gov.va.med.mhv.usermgmt.service.EntityMaintenanceService;
import gov.va.med.mhv.usermgmt.service.MviSyncCommErrServiceResponse;
import gov.va.med.mhv.usermgmt.service.PatientService;
import gov.va.med.mhv.usermgmt.service.PatientServiceResponse;
import gov.va.med.mhv.usermgmt.service.ServiceFactory;
import gov.va.med.mhv.usermgmt.service.UserProfileCollectionServiceResponse;
import gov.va.med.mhv.usermgmt.service.UserProfileDeactivationAuditCollectionServiceResponse;
import gov.va.med.mhv.usermgmt.service.UserProfileDeactivationReasonCollectionServiceResponse;
import gov.va.med.mhv.usermgmt.service.UserProfileDeactivationReasonServiceResponse;
import gov.va.med.mhv.usermgmt.service.UserProfileService;
import gov.va.med.mhv.usermgmt.service.UserProfileServiceResponse;
import gov.va.med.mhv.usermgmt.service.handler.MviProperties;
import gov.va.med.mhv.usermgmt.service.handler.VapiiProperties;
import gov.va.med.mhv.usermgmt.transfer.MviSyncCommErr;
import gov.va.med.mhv.usermgmt.transfer.Patient;
import gov.va.med.mhv.usermgmt.transfer.TermsConditionsVerHist;
import gov.va.med.mhv.usermgmt.transfer.TransferObjectFactory;
import gov.va.med.mhv.usermgmt.transfer.UserAcceptTermsHistory;
import gov.va.med.mhv.usermgmt.transfer.UserPasswordHistory;
import gov.va.med.mhv.usermgmt.transfer.UserProfile;
import gov.va.med.mhv.usermgmt.transfer.UserProfileDeactivationAudit;
import gov.va.med.mhv.usermgmt.transfer.UserProfileDeactivationReason;
import gov.va.med.mhv.usermgmt.transfer.UserProfilePK;
import gov.va.med.mhv.usermgmt.util.Auditor;
import gov.va.med.mhv.usermgmt.util.MessageKeys;
import gov.va.med.mhv.usermgmt.util.PasswordUtil;
import gov.va.med.mhv.usermgmt.util.PatientCorrelationStatusUtils;
import gov.va.med.mhv.usermgmt.util.UserProfileActivationStateUtils;
import gov.va.med.mhv.usermgmt.util.mvi.MviConstants;
import gov.va.www.vapiiPartnerServices.schema._1_1.FormType;
import gov.va.www.vapiiPartnerServices.schema._1_1.GetDocumentForUserRequest;
import gov.va.www.vapiiPartnerServices.schema._1_1.GetDocumentForUserResponse;
import gov.va.www.vapiiPartnerServices.schema._1_1.GetFormStatusForUserRequest;
import gov.va.www.vapiiPartnerServices.schema._1_1.GetFormStatusForUserResponse;
import gov.va.www.vapiiPartnerServices.schema._1_1.User;
import gov.va.www.vapiiPartnerServices.schema._1_1.UserIdType;
import gov.va.www.vapiiPartnerServices.schema._1_1.VapiiPartnerServicesProxy;

import java.sql.Timestamp;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.tigris.atlas.config.ConfigurationManager;
import org.tigris.atlas.messages.Message;
import org.tigris.atlas.service.BlobServiceResponse;
import org.tigris.atlas.service.BooleanServiceResponse;
import org.tigris.atlas.service.ServiceResponse;
import org.tigris.atlas.service.StringServiceResponse;
import org.tigris.atlas.service.TimestampServiceResponse;
import org.tigris.atlas.service.VoidServiceResponse;

/**
 * Service implementation class for the UserProfile service
 */
public class UserProfileServiceImpl extends MHVAbstractService
    implements UserProfileService
{

    private static final Log LOG = LogFactory.getLog(UserProfileServiceImpl.
        class);

	private static final Properties PROPS = ConfigurationManager.
        getConfiguration("/mailCfg.properties");

	private static final Properties PROPS_MPI = ConfigurationManager.
    getConfiguration("/mpi.properties");

	private static final String EMAIL_SUBJECT =
        "My HealtheVet Account Activity Update";

	private enum ENVTYPES { 
		DEV("dev"), INTB("intb"), SYSTEST("systest"), PREPROD("preprod"), PROD("prod"); 
		private String envValue;
		private ENVTYPES(String value) {
			this.envValue = value;
		}
		public String value(){
			return envValue;
		}
	};
	
	private String WILDCARD = "%";

    //MHV_CodeCR1299 - Made the auditing optional.
	public UserProfileServiceResponse saveUserProfile(UserProfile userProfile,
	        String verifySsn) {
		return saveUserProfile(userProfile, verifySsn, true);
	}

	/**
	 * Execute the SaveUserProfile service.
     * Saves a user profile and associates with a patient if applicable
	 *
     *
     * MHV_CodeCR1299 - Made the auditing optional.
     *
	 * @return SaveUserProfileServiceResponse
	 */
	public UserProfileServiceResponse saveUserProfile(UserProfile userProfile,
        String verifySsn, boolean doAudit) {
		boolean atRegistration = false;

		UserProfileServiceResponse response = new UserProfileServiceResponse();
		Long id = userProfile.getId();

		if(id == null) {
			atRegistration = true;
		}

		// Added for MVI
		MviProperties mviProp = MviProperties.getInstance();


		UserProfile savedUserProfile = doUserProfileSave(userProfile, verifySsn, doAudit);
		Long userId = savedUserProfile.getId();


		response.setUserProfile(savedUserProfile);
		if (savedUserProfile.getMessages().hasErrorMessages()) {
			return response;
		}


		if(mviProp.getIsMviEnabled()) {
			if (((userProfile.getIsPatient() == null) || !userProfile.getIsPatient())
					&& ((userProfile.getIsVeteran() == null) || !userProfile.getIsVeteran())
			) {
				if(id == null)
					addInfo(response, "mvi.basic.success.response",new String[]{MviConstants.BASIC_URL, MviConstants.TARGET_NEW_WINDOW});
				return response;
			}


		}
		else {
			if ((userProfile.getIsPatient() == null)
					|| !userProfile.getIsPatient().booleanValue()) {
				return response;
			}
		}

		// Check to see if there is already a patient for the user profile
		PatientService patientService = ServiceFactory.createPatientService();
		PatientServiceResponse patientForUserResponse = patientService.getPatientForUser(userProfile);
		Patient existPat = patientForUserResponse.getPatient();
		if (existPat != null && !existPat.getCorrelationStatus().equals(PatientCorrelationStatusUtils.UNCORRELATED)) {
		// Patient already associated
			return response;
		}
		else if (existPat != null){
			VoidServiceResponse voidServiceResponse =
				ServiceFactory.createMviAsyncIntegrationService().existingUserWithPendingDataMismatch(savedUserProfile);
			Iterator it = voidServiceResponse.getMessages().getInformationalMessages().iterator();
			while(it.hasNext()) {
				Message message = (Message)it.next();
				String msgKey = message.getKey();
				if(msgKey.equalsIgnoreCase("mvi.parse.response.ae.issue")) {
					addInfo(response, "mvi.parse.response.ae.issue",new String[]{MviConstants.ADVANCED_URL, MviConstants.TARGET_NEW_WINDOW, MviConstants.CONTACT_LOGGEDMHV_URL});
					return response;
				}
			}

			PatientService patientService2 = ServiceFactory.createPatientService();
			PatientServiceResponse patientForUserResponse2 = patientService2.getPatientForUser(userProfile);
			Patient existPat2 = patientForUserResponse2.getPatient();
			if(existPat2.getMatchedDateTime() != null) {
				addInfo(response, "mvi.update.advanced.success.response",new String[]{MviConstants.ADVANCED_URL, MviConstants.TARGET_NEW_WINDOW});
			}
			else if(existPat2.getCorrelationStatus().equals(PatientCorrelationStatusUtils.UNCORRELATED)) {
				addInfo(response, "mvi.update.advanced.failure.response",new String[]{MviConstants.ADVANCED_URL, MviConstants.TARGET_NEW_WINDOW, MviConstants.CONTACT_LOGGEDMHV_URL});
			}
			return response;
		}

		// No patient exists for the user profile in MHV.
        // Check the patient registry.
		String firstName = userProfile.getFirstName();
		String lastName  = userProfile.getLastName();
		String middleName  = userProfile.getMiddleName();
		String gender  = userProfile.getGender().getName();
		if(gender.equalsIgnoreCase("Male")){
			gender = "M";
		}else{
			gender = "F";
		}

		Date birthDate = userProfile.getBirthDate();
		String ssn = userProfile.getSsn();

		PatientServiceResponse patientRegistryResponse = null;

		/*
		 * Checking the property file to see if we need to call getPatientFromRegistryWithMiddleName or getPatientFromRegistry
		 */
		if(mviProp.getIsMviEnabled()) {
			patientRegistryResponse = patientService.getPatientFromMVI(savedUserProfile, atRegistration);
			if (patientRegistryResponse.getMessages().hasErrorMessages()) {
				response.getUserProfile().setId(id);
				copyMessages(response, patientRegistryResponse);
				return response;
			}
			Patient patient = patientRegistryResponse.getPatient();
			if(patient == null) {
				// Some issues with getting the Patient/Veteran
				//savedUserProfile.setIsPatient(false);
				//savedUserProfile.setIsVeteran(true);
				Iterator it = patientRegistryResponse.getMessages().getInformationalMessages().iterator();
				if(id == null) {
					copyMessages(response, patientRegistryResponse);
					//Auditor.auditMviEvent(savedUserProfile.getId(),ActivityActorTypeEnumeration.SELF,false, ActivityActionTypeEnumeration.UPGRADE_TO_ADVANCED, ActivityTypeEnumeration.REGISTRATION);
					Auditor.auditMviEvent(savedUserProfile.getId(),ActivityActorTypeEnumeration.SYSTEM,false, ActivityActionTypeEnumeration.UPGRADE_TO_ADVANCED, ActivityTypeEnumeration.ACCOUNT_UPGRADE);
				}
				else {
					//Auditor.auditMviEvent(savedUserProfile.getId(),ActivityActorTypeEnumeration.SELF,false, ActivityActionTypeEnumeration.UPGRADE_TO_ADVANCED, ActivityTypeEnumeration.PROFILES);
					Auditor.auditMviEvent(savedUserProfile.getId(),ActivityActorTypeEnumeration.SYSTEM,false, ActivityActionTypeEnumeration.UPGRADE_TO_ADVANCED, ActivityTypeEnumeration.ACCOUNT_UPGRADE);
					boolean parseErrorExists = false;
					boolean aeErrorUpdateUserProfile = false;
					while(it.hasNext()) {
						Message message = (Message)it.next();
						String msgKey = message.getKey();
						if(msgKey.equalsIgnoreCase("mvi.parse.response.issue")) {
							parseErrorExists = true;
						} else if(msgKey.equalsIgnoreCase("mvi.parse.response.ae.issue")) {
							aeErrorUpdateUserProfile = true;
						}
					}
					if(parseErrorExists) {
						addInfo(response, "mvi.update.parse.response.issue",new String[]{MviConstants.CONTACT_LOGGEDMHV_URL});
					}else if(aeErrorUpdateUserProfile) {
						addInfo(response, "mvi.parse.response.ae.issue",new String[]{MviConstants.BASIC_URL, MviConstants.TARGET_NEW_WINDOW} );
					}else {
						addInfo(response, "mvi.update.advanced.failure.response",new String[]{MviConstants.ADVANCED_URL, MviConstants.TARGET_NEW_WINDOW, MviConstants.CONTACT_LOGGEDMHV_URL});
					}
				}
				ServiceFactory.createEntityMaintenanceService().save(savedUserProfile);
				while(it.hasNext()) {
					Message message = (Message)it.next();
					String msgKey = message.getKey();
					if(msgKey.equalsIgnoreCase("mvi.parse.response.issue")) {
						MviSyncCommErrServiceResponse mviSyncCommErrSerRes = ServiceFactory.createMviSyncCommErrService().getRequiredMviSyncByProfileId(savedUserProfile.getId());
						MviSyncCommErr mviSyncCommError = mviSyncCommErrSerRes.getMviSyncCommErr();
						if(mviSyncCommError == null) {
							MviSyncCommErr mviSyncCommError2 = TransferObjectFactory.createMviSyncCommErr();
							mviSyncCommError2.setUserProfileId(savedUserProfile.getId());
							mviSyncCommError2.setCreatedTime(new Timestamp(System.currentTimeMillis()));
							ServiceFactory.createEntityMaintenanceService().save(mviSyncCommError2);
						}

					}
				}
				//copyMessages(response, patientRegistryResponse);
				return response;
			}
			else {
				patient.setUserProfile(savedUserProfile);
				patient.setCorrelationStatus(PatientCorrelationStatusUtils.MVI_DATA_MATCH);
				patient.setMatchedDateTime(new Timestamp(Calendar.getInstance().getTime().getTime()));
				//copyMessages(response, patientRegistryResponse);
				PatientServiceResponse psr = ServiceFactory.createEntityMaintenanceService().save(patient);
				if(id == null) {
					copyMessages(response, patientRegistryResponse);
					copyMessages(response, psr);
					//Auditor.auditMviEvent(savedUserProfile.getId(),ActivityActorTypeEnumeration.SELF,true, ActivityActionTypeEnumeration.UPGRADE_TO_ADVANCED, ActivityTypeEnumeration.REGISTRATION);
					Auditor.auditMviEvent(savedUserProfile.getId(),ActivityActorTypeEnumeration.SYSTEM,true, ActivityActionTypeEnumeration.UPGRADE_TO_ADVANCED, ActivityTypeEnumeration.ACCOUNT_UPGRADE);
				}

				else {
					//Auditor.auditMviEvent(savedUserProfile.getId(),ActivityActorTypeEnumeration.SELF,true, ActivityActionTypeEnumeration.UPGRADE_TO_ADVANCED, ActivityTypeEnumeration.PROFILES);
					Auditor.auditMviEvent(savedUserProfile.getId(),ActivityActorTypeEnumeration.SYSTEM,true, ActivityActionTypeEnumeration.UPGRADE_TO_ADVANCED, ActivityTypeEnumeration.ACCOUNT_UPGRADE);
					addInfo(response, "mvi.update.advanced.success.response",new String[]{MviConstants.ADVANCED_URL, MviConstants.TARGET_NEW_WINDOW});
				//	ServiceFactory.createMviAsyncIntegrationService().addPersonForCorrelation(patient, savedUserProfile, null);
				}
				/*
				if(patient.getFacilitys() != null)
					savedUserProfile.setIsPatient(true);
				if(savedUserProfile.getIsPatient() == true && patient.getFacilitys() == null) {
					savedUserProfile.setIsPatient(false);
					savedUserProfile.setIsVeteran(true);
				}
				*/
				savedUserProfile.setIsMPIControlled(true);
				ServiceFactory.createEntityMaintenanceService().save(savedUserProfile);
				//copyMessages(response, psr);
			}
		}
		else {
			if(!PROPS_MPI.isEmpty() && PROPS_MPI.getProperty("useMiddleNameAndGender", "true").equalsIgnoreCase("true")){
				if (LOG.isDebugEnabled()) {
					LOG.debug("Middle name: " + userProfile.getMiddleName()
							+ "gender: " + userProfile.getGender().getName()
							+ " for user id: " + userProfile.getId()
							+ " and useMiddleNameAndGender property value from mpi.properties: "
							+ PROPS_MPI.getProperty("useMiddleNameAndGender"));
				}
				patientRegistryResponse = patientService.
				getPatientFromRegistryWithMiddleName(firstName, lastName, middleName, gender, birthDate, ssn, userId);
			}else{
				if (LOG.isDebugEnabled()) {
					LOG.debug("In UserProfileServiceImpl::saveUserProfile::for user id: " + userProfile.getId() + " not calling middle name service");
				}
				patientRegistryResponse = patientService.
				getPatientFromRegistry(firstName, lastName, birthDate, ssn, userId);
			}
			Patient patient = patientRegistryResponse.getPatient();
			if (patientRegistryResponse.getMessages().hasErrorMessages()) {
				response.getMessages().addMessages(patientRegistryResponse.
						getMessages());
				//  If this is a new user profile, the id needs to be reset
				// since the txn will now roll back
				response.getUserProfile().setId(id);
				return response;
			}

			if (patient != null && patient.getAllMessages().hasErrorMessages()) {
				response.getMessages().addMessages(patient.getAllMessages());
			} else if (patient == null) {
				addError(response, MessageKeys.PATIENT_NOT_FOUND);
				//  If this is a new user profile, the id needs to be reset since
				// the txn will now roll back
				response.getUserProfile().setId(id);
			} else {
				// Patient was found in registry so associate it with the
				// user profile
				patient.setUserProfile(savedUserProfile);
				PatientServiceResponse psr = ServiceFactory.
				createEntityMaintenanceService().save(patient);
				copyMessages(response, psr);
			}

		}

		return response;
	}

	/**
	 * Execute the VerifyPasswordHints service
	 *
	 * @return VerifyPasswordHintsServiceResponse
	 */
	public BooleanServiceResponse verifyPasswordHints(UserProfile userProfile,
        String passwordHintAnswer1, String passwordHintAnswer2)
    {
		BooleanServiceResponse response = new BooleanServiceResponse();
		response.setValue(Boolean.TRUE);

		UserProfile up = ServiceFactory.createEntityMaintenanceService().
            findByPrimaryKey(userProfile.getUserProfilePK()).getUserProfile();
		if (up == null) {
			addError(response, UserManagementMessages.USER_NOT_FOUND);
			response.setBoolean(Boolean.FALSE);
			return response;
		}
		// Kumar,Vyasa started 06/01/2013
		if (StringUtils.isNotBlank(passwordHintAnswer1) && passwordHintAnswer1.endsWith("FP")) {
			if (StringUtils.isBlank(StringUtils.substringBefore(passwordHintAnswer1,"FP")) || StringUtils.isBlank(passwordHintAnswer2)) {
				//addError(response, UserManagementMessages.HINTS_ANSWER_REQUIRED);
				addError(response, UserManagementMessages.HINTS_ANSWER_ONE);
				addError(response, UserManagementMessages.HINTS_ANSWER_TWO);
				response.setBoolean(Boolean.FALSE);
				return response;
			}
		}
		if (StringUtils.isBlank(passwordHintAnswer1) || StringUtils.isBlank(passwordHintAnswer2)) {
				//addError(response, UserManagementMessages.HINTS_ANSWER_REQUIRED);
				addError(response, UserManagementMessages.HINTS_ANSWER_ONE);
				addError(response, UserManagementMessages.HINTS_ANSWER_TWO);
				response.setBoolean(Boolean.FALSE);
			return response;
		}
		// Kumar ended 06/01/2013

		if (! verifyHintAnswers(up, passwordHintAnswer1, passwordHintAnswer2)) {
			//addError(response, UserManagementMessages.INVALID_HINTS);
			addError(response, UserManagementMessages.HINTS_ANSWER_ONE);
			addError(response, UserManagementMessages.HINTS_ANSWER_TWO);
			response.setBoolean(Boolean.FALSE);
		}

		return response;
	}


	/**
	 * Execute the GetForgottenUserName service
	 *
	 * @return GetForgottenUserNameServiceResponse
	 */
	public StringServiceResponse getForgottenUserName(UserProfile userProfile,
        String passwordHintAnswer1, String passwordHintAnswer2)
    {
		String userEmail = userProfile.getContactInfoEmail();

		StringServiceResponse response = new StringServiceResponse();
		EntityMaintenanceService ems = ServiceFactory.
            createEntityMaintenanceService();
		UserProfile up = ems.findByPrimaryKey(userProfile.getUserProfilePK()).
            getUserProfile();

		// CR2161 and CR2162 Kumar,Vyasa started 06/01/2013
		if (StringUtils.isNotBlank(passwordHintAnswer1) && passwordHintAnswer1.endsWith("FP")) {
			if (StringUtils.isBlank(StringUtils.substringBefore(passwordHintAnswer1,"FP")) || StringUtils.isBlank(passwordHintAnswer2)) {
				//addError(response, UserManagementMessages.HINTS_ANSWER_REQUIRED);
				addError(response, UserManagementMessages.HINTS_ANSWER_ONE);
				addError(response, UserManagementMessages.HINTS_ANSWER_TWO);
				// response.setBoolean(Boolean.FALSE);
				return response;
			}
		}
		if (StringUtils.isBlank(passwordHintAnswer1) || StringUtils.isBlank(passwordHintAnswer2)) {
				//addError(response, UserManagementMessages.HINTS_ANSWER_REQUIRED);
				addError(response, UserManagementMessages.HINTS_ANSWER_ONE);
				addError(response, UserManagementMessages.HINTS_ANSWER_TWO);
		 		// response.setBoolean(Boolean.FALSE);
			   return response;
		}
     	//Kumar ended  06/01/2013

		if (verifyHintAnswers(up, passwordHintAnswer1, passwordHintAnswer2)) {
			response.setString(up.getUserName());

			/*
			 * CR2897: Sends out an email once the user changes the
             * password successfully and also if the user has an email
             * address in the profile
			 */
		    EmailServiceDelegate service = gov.va.med.mhv.core.service.
                delegate.ServiceDelegateFactory.createEmailServiceDelegate();

			if (!StringUtils.isBlank(userEmail)) {
				/*
				 * CR 3350; replyTo address has been changed and is now
                 * coming from email config properties file
				 */
				service.sendMessageWithReplyTo(PROPS.getProperty(
                    "userprofile.from.email"), EMAIL_SUBJECT, getMessage(),
                    userEmail.toString(), PROPS.getProperty(
                    "userprofile.replyTo.email"));
			}
		} else {
		//	addError(response, UserManagementMessages.INVALID_HINTS);
			addError(response, UserManagementMessages.HINTS_ANSWER_ONE);
			addError(response, UserManagementMessages.HINTS_ANSWER_TWO);
		}

		return response;
	}

	/*
	 *  CR 1160 - adding SM T&C info
	 *  Retrieve timestamp for SM T&C acceptance
	 *
	 */
	public TimestampServiceResponse getUserAcceptTermsHistByUserProfileId(Long id) {
		List userAcceptHistoryBOs = UserAcceptTermsHistoryBO.
        getUserAcceptTermsHistByProfileId(id);

		if(LOG.isDebugEnabled()){
			LOG.debug("SM T&C UserProfileServiceImpl.getUserAcceptTermsHistByUserProfileId() userAcceptHistoryBOs.size():" + userAcceptHistoryBOs.size());
		}

		TimestampServiceResponse response = new TimestampServiceResponse();

		if(userAcceptHistoryBOs != null && userAcceptHistoryBOs.size()>0) {
			UserAcceptTermsHistoryBO bo = (UserAcceptTermsHistoryBO)userAcceptHistoryBOs.get(0);
			if(LOG.isDebugEnabled()){
				LOG.debug("SM T&C UserProfileServiceImpl.getUserAcceptTermsHistByUserProfileId() bo.getUserUpdatedTime():" + bo.getUserUpdatedTime());
			}
			response.setTimestamp(bo.getUserUpdatedTime());
		}
		return response;
	}

	/*
	 * Builds an email body for forgot username portlet
	 */
	private static String getMessage() {
	    StringBuilder messageText = new StringBuilder();

	    messageText.append("This is a courtesy message to notify you of recent activity involving your My HealtheVet password or User ID." +
	    		           " If you did not recently change your password or request your User ID, please notify us by using the Contact MHV" +
	    		           " link provided on the My HealtheVet website- " + "www.myhealth.domain.");

	    messageText.append("\n\n ------------------------------------------------------- ");
	    messageText.append("\n **This is an automated message.  Please do not reply** ");
	    messageText.append("\n ------------------------------------------------------- \n\n");

	    messageText.append("This e-mail is intended only for the person or entity to which it is addressed, and may contain information that" +
	    		           " is privileged, confidential, or otherwise protected from disclosure.  Dissemination, distribution, or copying of" +
	    		           " this e-mail or the information herein by anyone other than the intended recipient or for official internal VHA " +
	    		           " and/or VA business is prohibited.");

	    messageText.append("\n\n");

	    return messageText.toString();
	  }

	/**
	 * Execute the getUserProfileForVAAFITraits service
	 *
	 * @return UserProfileServiceResponse
	 */
	public UserProfileServiceResponse getUserProfileForVAAFITraits(
		String firstName, String lastName, Date birthDate,
		GenderEnumeration gender)
	{
		UserProfileServiceResponse response = new UserProfileServiceResponse();

		String lowerFirstName = StringUtils.trimToNull(StringUtils.lowerCase(
			firstName));
		String lowerLastName = StringUtils.lowerCase(lastName);
		Collection profiles = UserProfileBO.getProfileByNameBirthDateGender(
			lowerFirstName, lowerLastName, birthDate, gender);
		profiles = UserProfileAssembler.getUserProfileCollection(profiles);

		if (profiles.isEmpty()) {
			addError(response, UserManagementMessages.USER_NOT_FOUND);
			return response;
		}

		UserProfile up = (UserProfile) profiles.iterator().next();
		response.setUserProfile(up);

		return response;
	}


	/**
	 * Execute the getPasswordHintsForForgottenUserName service
	 *
	 * @return GetPasswordHintsForForgottenUserNameServiceResponse
	 */
	public UserProfileServiceResponse getPasswordHintsForForgottenUserName(
        String firstName, String lastName, Date birthDate,
        GenderEnumeration gender)
    {
		UserProfileServiceResponse response = new UserProfileServiceResponse();

		String lowerFirstName = StringUtils.trimToNull(StringUtils.lowerCase(
			firstName));
		String lowerLastName = StringUtils.lowerCase(lastName);
		Collection profiles = UserProfileBO.getProfileByNameBirthDateGender(
            lowerFirstName, lowerLastName, birthDate, gender);
		profiles = UserProfileAssembler.getUserProfileCollection(profiles);

		if (profiles.isEmpty()) {
			addError(response, UserManagementMessages.USER_NOT_FOUND);
			return response;
		}

		UserProfile up = (UserProfile) profiles.iterator().next();

		//CCR# 3912: modify the sql to allow deactive users.
		// if deactivated, get the appropriate message so that user not found
		// and user account deactivated are differentiated.
		if (up.getDeactivationReason() != null)
		{
			addError(response, UserManagementMessages.ACCOUNT_DEACTIVATED);
			return response;
		}

		up.setPasswordHintAnswer1(null);
		up.setPasswordHintAnswer2(null);
		up.setUserName(null);

		response.setUserProfile(up);

		return response;
	}


	/**
	 * Execute the getPasswordHintsForForgottenPassword service
	 *
	 * @return GetPasswordHintsForForgottenPasswordServiceResponse
	 */
	public UserProfileServiceResponse getPasswordHintsForForgottenPassword(
        String userName, String firstName, String lastName, Date birthDate,
        GenderEnumeration gender)
    {
		UserProfileServiceResponse response = new UserProfileServiceResponse();

		String lowerFirstName = StringUtils.trimToNull(StringUtils.lowerCase(
			firstName));
		String lowerLastName = StringUtils.lowerCase(lastName);
		String lowerUserName = StringUtils.lowerCase(userName);
		Collection profiles = UserProfileBO.
            getProfileByNameBirthDateGenderUserName(lowerFirstName,
            lowerLastName, birthDate, gender, lowerUserName);
		profiles = UserProfileAssembler.getUserProfileCollection(profiles);

		if (profiles.isEmpty()) {
			addError(response, UserManagementMessages.USER_NOT_FOUND);
			return response;
		}

		UserProfile up = (UserProfile) profiles.iterator().next();

		//CCR# 3912: modify the sql to allow deactive users.
		// if deactivated, get the appropriate message so that user not found
		// and user account deactivated are differentiated.
		if (up.getDeactivationReason() != null)
		{
			addError(response, UserManagementMessages.ACCOUNT_DEACTIVATED);
			return response;
		}

		up.setPasswordHintAnswer1(null);
		up.setPasswordHintAnswer2(null);
		up.setUserName(null);

		response.setUserProfile(up);

		return response;
	}


	/**
	 * Execute the GetProfileForUser service
	 *
	 * @return GetProfileForUserServiceResponse
	 */
	public UserProfileServiceResponse getProfileForUser(String userName) {
		UserProfileServiceResponse response = new UserProfileServiceResponse();

		List l = UserProfileBO.getUserProfileByUserName(userName != null
            ? userName.toLowerCase() : null);
		if (l.size() > 0) {
			UserProfileBO upbo = (UserProfileBO) l.get(0);
			UserProfile up = new UserProfileAssembler().createUserProfile(upbo);
			response.setUserProfile(up);
		}

		return response;
	}

	/**
	 * Execute the GetUserProfileById service
	 *
	 * @return GetUserProfileByIdServiceResponse
	 */
	public UserProfileServiceResponse getUserProfileById(Long id) {
		UserProfileServiceResponse response = new UserProfileServiceResponse();

		List l = UserProfileBO.getUserProfileById(id);
		if (l.size() > 0) {
			UserProfileBO upbo = (UserProfileBO) l.get(0);
			UserProfile up = new UserProfileAssembler().createUserProfile(upbo);
			response.setUserProfile(up);
		}

		return response;
	}

	/**
	 * Execute the SearchUser service
	 *
	 * MHV_CodeCR1386 - Added ssn and dob as part of query.
	 * @return SearchUserServiceResponse
	 */
	public UserProfileCollectionServiceResponse searchPatientAndVet(String mhvId, String firstName, String lastName, String email, String userName,
		String ssn, Date birthDate) {

		UserProfileCollectionServiceResponse response =
            new UserProfileCollectionServiceResponse();

		String lowerFirstName = StringUtils.isNotBlank(firstName)
            ? StringUtils.lowerCase(firstName) + WILDCARD : WILDCARD;
		String lowerLastName = StringUtils.isNotBlank(lastName)
            ? StringUtils.lowerCase(lastName) + WILDCARD : WILDCARD;
		String lowerEmail = StringUtils.isNotBlank(email)
            ? StringUtils.lowerCase(email) + WILDCARD : WILDCARD;
		String lowerMhvId = StringUtils.isNotBlank(mhvId)
            ? StringUtils.lowerCase(mhvId) + WILDCARD : WILDCARD;
		String lowerUserName = StringUtils.isNotBlank(userName)
            ? StringUtils.lowerCase(userName) + WILDCARD : WILDCARD;
    	String lowerSsn = StringUtils.isNotBlank(ssn)
            ? WILDCARD + StringUtils.lowerCase(ssn) + WILDCARD : WILDCARD;

		// search only if validation passed
		if (!response.getMessages().hasErrorMessages()) {
            Collection profileBOs = null;
            if (StringUtils.isBlank(email)) {
            	if(birthDate != null) {
            		profileBOs = UserProfileBO.getPatientVetProfileByNameMhvIdUserNameWithBirthDate(
                            lowerFirstName, lowerLastName, lowerMhvId, lowerUserName, lowerSsn, birthDate);
            	} else {
            		profileBOs = UserProfileBO.getPatientVetProfileByNameMhvIdUserName(
                    lowerFirstName, lowerLastName, lowerMhvId, lowerUserName, lowerSsn);
            	}
            } else {
            	if(birthDate != null) {
	                profileBOs = UserProfileBO.getPatientVetProfileByNameEmailMhvIdUserName(
	                    lowerFirstName, lowerLastName, lowerEmail, lowerMhvId,
	                    lowerUserName, lowerSsn);
            	} else {
            		profileBOs = UserProfileBO.getPatientVetProfileByNameEmailMhvIdUserNameWithBirthDate(
                    lowerFirstName, lowerLastName, lowerEmail, lowerMhvId, lowerUserName, lowerSsn, birthDate);
            	}
            }

			Collection result = UserProfileAssembler.
                getUserProfileCollection(profileBOs);
			response.addUserProfiles(result);

			if (result.isEmpty()) {
				addInfo(response, UserManagementMessages.NO_USERS_FOUND);
            }
		}

		return response;
	}

	/**
	 * Execute the SearchUser service
	 *
	 * MHV_CodeCR1386 - Added ssn and dob as part of query.
	 * @return SearchUserServiceResponse
	 */
	public UserProfileCollectionServiceResponse searchUser(String mhvId, String firstName, String lastName, String email, String userName,
		String ssn, Date birthDate) {

		UserProfileCollectionServiceResponse response =
            new UserProfileCollectionServiceResponse();

		String lowerFirstName = StringUtils.isNotBlank(firstName)
            ? StringUtils.lowerCase(firstName) + WILDCARD : WILDCARD;
		String lowerLastName = StringUtils.isNotBlank(lastName)
            ? StringUtils.lowerCase(lastName) + WILDCARD : WILDCARD;
		String lowerEmail = StringUtils.isNotBlank(email)
            ? StringUtils.lowerCase(email) + WILDCARD : WILDCARD;
		String lowerMhvId = StringUtils.isNotBlank(mhvId)
            ? StringUtils.lowerCase(mhvId) + WILDCARD : WILDCARD;
		String lowerUserName = StringUtils.isNotBlank(userName)
            ? StringUtils.lowerCase(userName) + WILDCARD : WILDCARD;
    	String lowerSsn = StringUtils.isNotBlank(ssn)
            ? WILDCARD + StringUtils.lowerCase(ssn) + WILDCARD : WILDCARD;

		// search only if validation passed
		if (!response.getMessages().hasErrorMessages()) {
            Collection profileBOs = null;
            if (StringUtils.isBlank(email)) {
            	if(birthDate != null) {
            		profileBOs = UserProfileBO.getProfileByNameMhvIdUserNameWithBirthDate(
                            lowerFirstName, lowerLastName, lowerMhvId, lowerUserName, lowerSsn, birthDate);
            	} else {
            		profileBOs = UserProfileBO.getProfileByNameMhvIdUserName(
                    lowerFirstName, lowerLastName, lowerMhvId, lowerUserName, lowerSsn);
            	}
            } else {
            	//MHV_CodeCR1514 - US12.4 MVI Compliance Implementation - Fixed the bug where the methods are reversed.
            	if(birthDate != null) {
            		profileBOs = UserProfileBO.getProfileByNameEmailMhvIdUserNameWithBirthDate(
                            lowerFirstName, lowerLastName, lowerEmail, lowerMhvId, lowerUserName, lowerSsn, birthDate);
            	} else {
	                profileBOs = UserProfileBO.getProfileByNameEmailMhvIdUserName(
		                    lowerFirstName, lowerLastName, lowerEmail, lowerMhvId,
		                    lowerUserName, lowerSsn);
            	}
            }

			Collection result = UserProfileAssembler.
                getUserProfileCollection(profileBOs);
			response.addUserProfiles(result);

			if (result.isEmpty()) {
				addInfo(response, UserManagementMessages.NO_USERS_FOUND);
            }
		}

		return response;
	}

	//JAZZ:Defect# 19718 - Include middle name in the basic search.
	public UserProfileCollectionServiceResponse findUser(String mhvId, String firstName, String middleName,
			String lastName, String email, String userName, String ssn, Date birthDate) {

		UserProfileCollectionServiceResponse response =
            new UserProfileCollectionServiceResponse();

		String lowerFirstName = StringUtils.isNotBlank(firstName)
            ? StringUtils.lowerCase(firstName) + WILDCARD : WILDCARD;
    	String lowerMiddleName = StringUtils.isNotBlank(middleName)
            ? StringUtils.lowerCase(middleName) + WILDCARD : WILDCARD;
		String lowerLastName = StringUtils.isNotBlank(lastName)
            ? StringUtils.lowerCase(lastName) + WILDCARD : WILDCARD;
		String lowerEmail = StringUtils.isNotBlank(email)
            ? StringUtils.lowerCase(email) + WILDCARD : WILDCARD;
		String lowerMhvId = StringUtils.isNotBlank(mhvId)
            ? StringUtils.lowerCase(mhvId) + WILDCARD : WILDCARD;
		String lowerUserName = StringUtils.isNotBlank(userName)
            ? StringUtils.lowerCase(userName) + WILDCARD : WILDCARD;
    	String lowerSsn = StringUtils.isNotBlank(ssn)
            ? WILDCARD + StringUtils.lowerCase(ssn) + WILDCARD : WILDCARD;

		// search only if validation passed
		if (!response.getMessages().hasErrorMessages()) {
            Collection profileBOs = null;
            if (StringUtils.isBlank(email)) {
            	if(birthDate != null) {
            		profileBOs = UserProfileBO.getUserByNameMhvIdUserNameWithBirthDate(
                            lowerFirstName, lowerMiddleName, lowerLastName, lowerMhvId, lowerUserName, lowerSsn, birthDate);
            	} else {
            		profileBOs = UserProfileBO.getUserByNameMhvIdUserName(
                    lowerFirstName, lowerMiddleName, lowerLastName, lowerMhvId, lowerUserName, lowerSsn);
            	}
            } else {
            	//MHV_CodeCR1514 - US12.4 MVI Compliance Implementation - Fixed the bug where the methods are reversed.
            	if(birthDate != null) {
            		profileBOs = UserProfileBO.getUserByNameEmailMhvIdUserNameWithBirthDate(
                            lowerFirstName, lowerMiddleName, lowerLastName, lowerEmail, lowerMhvId, lowerUserName, lowerSsn, birthDate);
            	} else {
	                profileBOs = UserProfileBO.getUserByNameEmailMhvIdUserName(
		                    lowerFirstName, lowerMiddleName, lowerLastName, lowerEmail, lowerMhvId,
		                    lowerUserName, lowerSsn);
            	}
            }

			Collection result = UserProfileAssembler.
                getUserProfileCollection(profileBOs);
			response.addUserProfiles(result);

			if (result.isEmpty()) {
				addInfo(response, UserManagementMessages.NO_USERS_FOUND);
            }
		}

		return response;
	}

	/**
	 * @see gov.va.med.mhv.usermgmt.service.UserProfileService#acceptTerms(
     *      gov.va.med.mhv.usermgmt.transfer.UserProfile,java.sql.Timestamp,
     *      java.sql.Timestamp)
	 */
	public VoidServiceResponse acceptTerms(UserProfile userProfile,
        Timestamp termsModifiedTime,Timestamp termsAcceptedTime)
    {
	  VoidServiceResponse response = new VoidServiceResponse();

	  if (LOG.isDebugEnabled()) {
			LOG.debug("Before acceptTerms DB update" + userProfile.getUserName() + ":userProfile.getAcceptTerms()"+userProfile.getAcceptTerms());
			LOG.debug("Before acceptTerms DB update" + userProfile.getUserName() + ":userProfile.getAcceptSMTerms()"+userProfile.getAcceptSMTerms());
	  }

	  if (termsModifiedTime != null) {
		UserAcceptTermsHistoryBO userTermsBo;

		Collection userAcceptHistoryBOs = UserAcceptTermsHistoryBO.
            getUserAcceptTermsHistByUpdateTime(userProfile.getId(),
            termsModifiedTime);
		if ((userAcceptHistoryBOs == null)  ||userAcceptHistoryBOs.isEmpty()) {
            //does not have an answer yet
			TermsConditionsVerHist verHist;

			Collection versionHistoryBOs=TermsConditionsVerHistBO.
                getVersionByUpdateTime(termsModifiedTime);
			if ((versionHistoryBOs == null) || versionHistoryBOs.isEmpty()) {
                //this version has no record in the version table yet
				verHist = new TermsConditionsVerHist();
				verHist.setUpdatedTime(termsModifiedTime);
				TermsConditionsVerHistBO tvBo=BusinessObjectFactory.
                    createTermsConditionsVerHistBO();
				tvBo.setTermsConditionsVerHistValues(verHist);
				tvBo.save();
				verHist=tvBo.getTermsConditionsVerHistValues();
    		} else { //this version has a record
				verHist=((TermsConditionsVerHistBO)versionHistoryBOs.iterator().
                    next()).getTermsConditionsVerHistValues();
			}
			UserAcceptTermsHistory history=new UserAcceptTermsHistory();
			history.setUserProfileId(userProfile.getId());
			history.setAccepted(Boolean.TRUE);
			history.setUserUpdatedTime(termsAcceptedTime);
			history.setTermsConditionsVerHist(verHist);

			userTermsBo=BusinessObjectFactory.createUserAcceptTermsHistoryBO();
			userTermsBo.setUserAcceptTermsHistoryValues(history);
		} else{ //just update the answer
			userTermsBo =(UserAcceptTermsHistoryBO)userAcceptHistoryBOs.
                iterator().next();
			userTermsBo.setAccepted(new Boolean(true));
			userTermsBo.setUserUpdatedTime(termsAcceptedTime);

		    if (LOG.isDebugEnabled()) {
				LOG.debug("userProfile.getUserName():" + userProfile.getAcceptTerms()+ "entered a new userTermsHistory for " + termsAcceptedTime);
		   }

		}
		userTermsBo.save();
		if (userTermsBo.getMessages().getErrorMessages() != null){
			response.getMessages().addMessages(userTermsBo.getMessages());
		}
		//CR 1160 - Secured Message Terms and Conditions Implementation
		//Saving SM T&C for re-acceptance case.
		UserProfileServiceResponse upsResponse = getUserProfileById(userProfile.getId());
		UserProfile newUP = upsResponse.getUserProfile();
		if(userProfile.getAcceptSMTerms() != null) {
			newUP.setAcceptSMTerms(userProfile.getAcceptSMTerms());
			newUP.setAcceptTerms(true);
		    //MHV_CodeCR1299 - Made sure that audit won't happen
			//saveUserProfile(newUP, newUP.getSsn(), false); Commenting out so that saveUserProfile(UserProfile, String, boolean) doesn't get called twice.
			ServiceFactory.createEntityMaintenanceService().save(newUP);

		}

	  }

	  if (LOG.isDebugEnabled()) {
			LOG.debug("After acceptTerms DB update" + userProfile.getUserName() + ":userProfile.getAcceptTerms()"+userProfile.getAcceptTerms());
			LOG.debug("After acceptTerms DB update" + userProfile.getUserName() + ":userProfile.getAcceptSMTerms()"+userProfile.getAcceptSMTerms());
	  }

	  return response;
	}

	/**
	 * @see gov.va.med.mhv.usermgmt.service.UserProfileService#rejectTerms(
     *      gov.va.med.mhv.usermgmt.transfer.UserProfile)
	 */
	public VoidServiceResponse rejectTerms(UserProfile userProfile,
        Timestamp termsModifiedTime,Timestamp termsRejectedTime)
    {
		VoidServiceResponse response = new VoidServiceResponse();

		UserAcceptTermsHistoryBO userTermsBo = null;
		UserAcceptTermsHistory history = null;
		TermsConditionsVerHist verHist = null;

		Collection userAcceptHistoryBOs = UserAcceptTermsHistoryBO.
            getUserAcceptTermsHistByUpdateTime(userProfile.getId(),
            termsModifiedTime);
		if ((userAcceptHistoryBOs == null) || userAcceptHistoryBOs.isEmpty()) {
            //does not have an answer yet
			userTermsBo=BusinessObjectFactory.createUserAcceptTermsHistoryBO();
			history=new UserAcceptTermsHistory();
			Collection versionHistoryBOs = TermsConditionsVerHistBO.
                getVersionByUpdateTime(termsModifiedTime);
			if ((versionHistoryBOs == null) || versionHistoryBOs.isEmpty()) {
                //this version has no record in the version table yet
				verHist=new TermsConditionsVerHist();
				verHist.setUpdatedTime(termsModifiedTime);
				TermsConditionsVerHistBO tvBo = BusinessObjectFactory.
                    createTermsConditionsVerHistBO();
				tvBo.setTermsConditionsVerHistValues(verHist);
				tvBo.save();
				verHist=tvBo.getTermsConditionsVerHistValues();
    		} else{ //this version has a record
				verHist=((TermsConditionsVerHistBO) versionHistoryBOs.
                    iterator().next()).getTermsConditionsVerHistValues();
			}
			history.setUserProfileId(userProfile.getId());
			history.setAccepted(new Boolean(false));
			history.setUserUpdatedTime(termsRejectedTime);
			history.setTermsConditionsVerHist(verHist);
			userTermsBo.setUserAcceptTermsHistoryValues(history);
		} else{ //just update the answer
			userTermsBo = (UserAcceptTermsHistoryBO) userAcceptHistoryBOs.
                iterator().next();
			userTermsBo.setAccepted(new Boolean(false));
			userTermsBo.setUserUpdatedTime(termsRejectedTime);

		}

		userTermsBo.save();
		if (userTermsBo.getMessages().getErrorMessages() != null){
			response.getMessages().addMessages(userTermsBo.getMessages());
		}

		return response;
	}

	/**
	 * @see gov.va.med.mhv.usermgmt.service.UserProfileService#
     *  verifyLatestTermsAccepted(java.lang.String)
	 */
	public BooleanServiceResponse verifyLatestTermsAccepted(
        UserProfile userProfile, Timestamp latestTermsTime)
    {
		BooleanServiceResponse response = new BooleanServiceResponse();
		Collection userAcceptHistoryBOs=UserAcceptTermsHistoryBO.
            getUserAcceptTermsHistByUpdateTime(userProfile.getId(),
            latestTermsTime);
		if ((userAcceptHistoryBOs == null) || userAcceptHistoryBOs.isEmpty()) {

			response.setValue(Boolean.FALSE);
		} else if (((UserAcceptTermsHistoryBO)userAcceptHistoryBOs.iterator().
            next()).getAccepted().booleanValue())
        {
			response.setValue(Boolean.TRUE);
		} else {
			response.setValue(Boolean.FALSE);
		}

		return response;
	}

	/**
	 * Save the history of password, to reinforce the rule that user can not
     * reuse the last three passwords
	 * @param userProfile
	 * @param password
	 * @return
	 */
	public VoidServiceResponse savePasswordHistory(UserProfile userProfile,
        String password, Boolean tempFlag)
	{
		VoidServiceResponse response = new VoidServiceResponse();
		UserPasswordHistoryBO bo = BusinessObjectFactory.
            createUserPasswordHistoryBO();
		UserPasswordHistory hist=new UserPasswordHistory();
		hist.setPassword(PasswordUtil.digest(password, "SHA"));
		hist.setTempPasswordFlag(tempFlag);
		hist.setUpdateTime(new Timestamp(System.currentTimeMillis()));
		hist.setUserProfileId(userProfile.getId());
		bo.setUserPasswordHistoryValues(hist);
		bo.save();
		return response;
	}

	/**
	 * Verify if the password is the same as the previous three passwords
	 * @param userProfile
	 * @param password
	 * @return
	 */
	public BooleanServiceResponse hasPasswordBeenUsed(UserProfile userProfile,
        String password)
    {
		boolean used = false;
		BooleanServiceResponse response = new BooleanServiceResponse();
		Collection bos = UserPasswordHistoryBO.
            getPasswordHistoryByUserProfileId(userProfile.getId());
		String pwHash = PasswordUtil.digest(password, "SHA");
		Iterator i = bos.iterator();
		for (int count = 1; (count <= 3) && i.hasNext(); count++) {
			UserPasswordHistoryBO bo = (UserPasswordHistoryBO) i.next();
			if (bo.getPassword().equals(pwHash)){
				used = true;
				break;
			}
		}

		response.setBoolean(new Boolean(used));
		return response;
	}

	public BooleanServiceResponse isCurrentPasswordTemporary(
        UserProfile userProfile)
    {
		BooleanServiceResponse response = new BooleanServiceResponse();
		boolean tempPassword = false;
		Collection bos = UserPasswordHistoryBO.isCurrentPasswordTemporary(
            userProfile.getId()); //hist should either be empty or one element
		if (bos.iterator().hasNext()){
			UserPasswordHistoryBO bo = (UserPasswordHistoryBO) bos.iterator().
                next();
			tempPassword=bo.getTempPasswordFlag().booleanValue();
		}

		response.setBoolean(tempPassword);
		return response;
	}

    public VoidServiceResponse changeActivationState(UserProfile userProfile,
        UserProfileDeactivationReason reason, String changedBy, String notes, Date deathDate)
    {

	  //Admin Portal_CodeCR1959 - React/Deact Impl
       Precondition.assertNotNull("userProfile", userProfile);
       String originalLastName = userProfile.getLastName();

       VoidServiceResponse response = new VoidServiceResponse ();
       if (reason != null) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Deactivating account '" + userProfile.getUserName()
                    + ", because " + reason.getName() + ". " + StringUtils.
                    defaultString(notes));

            }
            if(reason.getName().equals("Deceased")) {
            	userProfile.setLastName(userProfile.getLastName()+"DECEASED");
            } else {
            	//US12.5 - Make sure to add "DELETE"
            	userProfile.setLastName(userProfile.getLastName()+"DELETED");
            }
            userProfile.setDeactivationReason(reason);
            userProfile.setDeathDate(deathDate);
            //JAZZ: 19292 - Take out matched date, set mpi controlled flag to null during deactivation.
            userProfile.setIsMPIControlled(false);
       } else {
   			addError(response, "mvi.helpdesk.devalidate.error.reason.missing", new String [] {"deactivateReason"});
            if (LOG.isDebugEnabled()) {
                LOG.debug("No deactivation reason for user:" + userProfile.getUserName());

            }
       }

       if ((notes == null)||(notes.length()==0)) {
    	   addError(response, "mvi.helpdesk.devalidate.error.description.missing", new String [] {"deactivateNote"});
       }
       	//Admin Portal_CodeCR1959 - Deact/React Impl
       if (!hasErrorMessages(response)) {
    	   UserProfile savedUserProfile = doUserProfileActivateSave(userProfile, userProfile.getSsn(), false);
           if (hasErrorMessage(savedUserProfile)) {
               userProfile.setLastName(originalLastName);
               copyMessages(response, savedUserProfile);
        	   Auditor.auditAccountDeactivateEvent(userProfile.getId(), ActivityActorTypeEnumeration.HELP_DESK_ADMINISTRATOR, false);
           } else {
        	   //Note this audit is only for deact/react and goes to user_profile_deact_audit table; different from aal table.
               auditActivationStateChange(response, userProfile, reason, changedBy, notes);
        	   Auditor.auditAccountDeactivateEvent(userProfile.getId(), ActivityActorTypeEnumeration.HELP_DESK_ADMINISTRATOR, true);
        	   //JAZZ:Task#20852 - Take out entry from eauthhash during deactivation for the user
        	   deleteHashByUserProfileId(userProfile.getId());
           }
       } else {
           userProfile.setLastName(originalLastName);
    	   Auditor.auditAccountDeactivateEvent(userProfile.getId(), ActivityActorTypeEnumeration.HELP_DESK_ADMINISTRATOR, false);
       }

       return response;
    }

	private String getFullName(UserProfile userProfile) {
		String firstName = userProfile.getFirstName();
		String middleName = userProfile.getMiddleName();
		String lastName = userProfile.getLastName();

		StringBuffer buff = new StringBuffer();
		buff.append(firstName);
		if (middleName != null) {
			buff.append(' ');
			buff.append(middleName);
		}
		buff.append(' ');
		buff.append(lastName);

		return(buff.toString());
	}

	private boolean duplicateCheck(UserProfile userProfile) {
		String firstName = StringUtils.lowerCase(userProfile.getFirstName());
		String lastName = StringUtils.lowerCase(userProfile.getLastName());

		if (UserProfileBO.getProfileByNameBirthDateSsnExcludeId(firstName, lastName,
				userProfile.getBirthDate(), userProfile.getSsn(), userProfile.getId()).size() > 1) {
			if (LOG.isDebugEnabled())
				LOG.debug("MviIntegrationServiceImpl.duplicateCheck(): Duplicate account found for the user: " + userProfile.getUserName());
		    return true;
        }
		return false;
	}

    //Admin Portal_CodeCR1959 - React/Deact Impl
    public VoidServiceResponse reactivateUserInHelpDesk(UserProfile userProfile,
            UserProfileDeactivationReason reason, String changedBy, String notes)
    {
        Precondition.assertNotNull("userProfile", userProfile);
        Precondition.assertNotNull("changedBy", changedBy);
        VoidServiceResponse response = new VoidServiceResponse();
        if (reason != null) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Deactivating account '" + userProfile.getUserName()
                    + ", because " + reason.getName() + ". " + StringUtils.
                    defaultString(notes));

            }
        } else {
   			addError(response, "mvi.helpdesk.devalidate.error.reason.missing", new String [] {"reactivateReason"});
            if (LOG.isDebugEnabled()) {
                LOG.debug("No deactivation reason for user:" + userProfile.getUserName());

            }
        }

        if ((notes == null)||(notes.length()==0)) {
    	   addError(response, "mvi.helpdesk.devalidate.error.description.missing", new String [] {"reactivateNote"});
        }

        if (!hasErrorMessages(response)) {
        	//US12.5 - Make sure to delete "DELETE"
            userProfile.setLastName(userProfile.getLastName().replaceAll("DELETED", ""));
            userProfile.setLastName(userProfile.getLastName().replaceAll("DECEASED", ""));

            //JAZZ:19556 - Make sure that the correct message display when there is a duplicate user
            if(duplicateCheck(userProfile)) {
            	addError(response, "mvi.helpdesk.revalidate.fail", new Object[] {getFullName(userProfile)});
		        Auditor.auditAccountReactivateEvent(userProfile.getId(), ActivityActorTypeEnumeration.HELP_DESK_ADMINISTRATOR, false);
		        return response;
            }

            userProfile.setDeactivationReason(null);
            userProfile.setDeathDate(null);
	        userProfile = doUserProfileReactivateSave(userProfile, userProfile.getSsn());
	        if (hasErrorMessage(userProfile)) {
	        	copyMessages(response, userProfile);
		        Auditor.auditAccountReactivateEvent(userProfile.getId(), ActivityActorTypeEnumeration.HELP_DESK_ADMINISTRATOR, false);
	        } else {
	        	//Note this audit is only for deact/react and goes to user_profile_deact_audit table; different from aal table.
	        	auditReactivationChange(response, userProfile, reason, changedBy, notes);
	        	Auditor.auditAccountReactivateEvent(userProfile.getId(), ActivityActorTypeEnumeration.HELP_DESK_ADMINISTRATOR, true);
	        }
        } else {
	        Auditor.auditAccountReactivateEvent(userProfile.getId(), ActivityActorTypeEnumeration.HELP_DESK_ADMINISTRATOR, false);
        }
        return response;
    }

    //Admin Portal_CodeCR1959 - React/Deact Impl
    private void auditReactivationChange(ServiceResponse response,
        UserProfile userProfile, UserProfileDeactivationReason reason,
        String changedBy, String notes)
    {
        UserProfileDeactivationAudit audit = TransferObjectFactory.
            createUserProfileDeactivationAudit();
        audit.setAction(UserProfileActivationStateUtils.ACTIVATED);
        audit.setUserProfile(userProfile);
        audit.setDeactivationReason(reason);
        audit.setChangeDate(TimestampUtils.createCurrentTime());
        audit.setChangedBy(changedBy);
        audit.setNotes(notes);
        copyMessages(response, ServiceFactory.createEntityMaintenanceService().
            save(audit));
    }

	//Admin Portal_CodeCR1959 - React/Deact Impl
    private void auditActivationStateChange(ServiceResponse response,
            UserProfile userProfile, UserProfileDeactivationReason reason,
            String changedBy, String notes)
        {
    		UserProfileDeactivationAudit audit = TransferObjectFactory.
                createUserProfileDeactivationAudit();
            audit.setAction((reason != null)
                ? UserProfileActivationStateUtils.DEACTIVATED
                : UserProfileActivationStateUtils.ACTIVATED);
            audit.setUserProfile(userProfile);
            audit.setDeactivationReason(reason);
            audit.setChangeDate(TimestampUtils.createCurrentTime());
            audit.setChangedBy(changedBy);
            audit.setNotes(notes);
            copyMessages(response, ServiceFactory.createEntityMaintenanceService().
                save(audit));
        }

    /* (non-Javadoc)
     * @see gov.va.med.mhv.usermgmt.service.UserProfileService#
     * findAllDeactivationReasons()
     */
    public UserProfileDeactivationReasonCollectionServiceResponse
        findAllDeactivationReasons()
    {
        UserProfileDeactivationReasonCollectionServiceResponse response =
            new UserProfileDeactivationReasonCollectionServiceResponse();
        List reasons = UserProfileDeactivationReasonBO.
            findAllUserProfileDeactivationReasons();
        if ((reasons != null) && !reasons.isEmpty()) {
            response.addItems(UserProfileDeactivationReasonAssembler.
                getUserProfileDeactivationReasonCollection(reasons));
        }
        return response;
    }

    /* (non-Javadoc)
     * @see gov.va.med.mhv.usermgmt.service.UserProfileService#
     * findDeactivationReason(java.lang.String)
     */
    public UserProfileDeactivationReasonServiceResponse
        findDeactivationReason(String name)
    {
        UserProfileDeactivationReasonServiceResponse response =
            new UserProfileDeactivationReasonServiceResponse();
        List reasons = UserProfileDeactivationReasonBO.
            findUserProfileDeactivationReasonByName(name);
        if ((reasons != null) && !reasons.isEmpty()) {
            response.setUserProfileDeactivationReason(
                ((UserProfileDeactivationReasonBO) reasons.get(0)).
                getUserProfileDeactivationReasonValues());
        }
        return response;
    }

    public UserProfileDeactivationAuditCollectionServiceResponse
        findDeactivationAudits(UserProfile userProfile)
    {
        Precondition.assertNotNull("userProfile", userProfile);
        Precondition.assertNotNull("userProfile.id", userProfile.getId());
        UserProfileDeactivationAuditCollectionServiceResponse response =
            new UserProfileDeactivationAuditCollectionServiceResponse();
        List audits = UserProfileDeactivationAuditBO.
            findUserProfileDeactivationAuditByUser(userProfile.getId());
        if ((audits != null) && !audits.isEmpty()) {
            response.addUserProfileDeactivationAudits(
                UserProfileDeactivationAuditAssembler.
                getUserProfileDeactivationAuditCollection(audits));
        }
        return response;
    }

    /*
     * MHV_CodeCR1299 - Made the auditing optional.
     */
    private UserProfile doUserProfileSave(UserProfile entity,
        String verifySsn) {
    	return doUserProfileSave(entity,verifySsn, true);
    }

    /*
     * Admin Portal_CodeCR1959 - React/Deact Impl
     */
    private UserProfile doUserProfileReactivateSave(UserProfile entity, String verifySsn) {
        UserProfileBO bo = null;
        if (entity != null) {
            UserProfilePK pk = entity.getUserProfilePK();
            if (pk.getValue() != null) {
                bo = UserProfileBO.findByPrimaryKey(pk);
            } else {
                bo = BusinessObjectFactory.createUserProfileBO();
                // Since this is a new entity, set the crate date to now.
                entity.setCreatedDate(new Timestamp(System.
                    currentTimeMillis()));
            }

            bo.setUserProfileValuesForReactivation(entity);
            bo.setDeathDate(null);
            bo.validate();
        	//CR2161 and CR2162 Kumar,Vyasa started here
            //bo.validatePasswordHintAnswers();
        	//CR2161 and CR2162 Kumar,Vyasa ended here
            bo.validateMatchingSSN(verifySsn);
            if (!bo.getMessages().hasErrorMessages()) {
                bo.save();
            } else {
            	Iterator iterator = bo.getMessages().getErrorMessages().iterator();
            	while(iterator.hasNext()) {
            		StringBuffer str = new StringBuffer("doUserProfileReactivateSave() failed for user profile id: "+entity.getId() + " ERROR:");
            		Message message = (Message)iterator.next();
        			str.append(message.getKey());
        			if (message.getInserts().size() > 0) {
        				str.append("; inserts=[").append(message.getInserts()).
        					append("]");
        			}
        			if (message.getProperties().size() > 0) {
        				str.append("; properties=[").append(message.getProperties()).
        					append("]");
        			}
        			if(LOG.isErrorEnabled()) {
        				LOG.error(str);
        			}
            	}
            }
        }
        return bo.getUserProfileValues();
    }
 private UserProfile doUserProfileActivateSave(UserProfile entity,
            String verifySsn, boolean doAudit) {
        UserProfileBO bo = null;
        if (entity != null) {
            UserProfilePK pk = entity.getUserProfilePK();
            if (pk.getValue() != null) {
                bo = UserProfileBO.findByPrimaryKey(pk);
                if(doAudit) {
                	Auditor.auditUpdateProfileEvent(bo.getId(),
                    ActivityActorTypeEnumeration.SELF);
                }
            } else {
                bo = BusinessObjectFactory.createUserProfileBO();
                // Since this is a new entity, set the crate date to now.
                entity.setCreatedDate(new Timestamp(System.
                    currentTimeMillis()));
            }

            bo.setUserProfileValues(entity);
            bo.validate();
        	//CR2161 and CR2162 Kumar,Vyasa started here
//            bo.validatePasswordHintAnswers();
        	//CR2161 and CR2162 Kumar,Vyasa ended here
            bo.validateMatchingSSN(verifySsn);
            if (!bo.getMessages().hasErrorMessages()) {
                bo.save();
            }
        }
        return bo.getUserProfileValues();
    }

    private UserProfile doUserProfileSave(UserProfile entity,
            String verifySsn, boolean doAudit) {
        UserProfileBO bo = null;
        if (entity != null) {
            UserProfilePK pk = entity.getUserProfilePK();
            if (pk.getValue() != null) {
                bo = UserProfileBO.findByPrimaryKey(pk);
                if(doAudit) {
                	Auditor.auditUpdateProfileEvent(bo.getId(),
                    ActivityActorTypeEnumeration.SELF);
                }
            } else {
                bo = BusinessObjectFactory.createUserProfileBO();
                // Since this is a new entity, set the crate date to now.
                entity.setCreatedDate(new Timestamp(System.
                    currentTimeMillis()));
            }

            bo.setUserProfileValues(entity);
            bo.validate();
        	//CR2161 and CR2162 Kumar,Vyasa started here
            bo.validatePasswordHintAnswers();
        	//CR2161 and CR2162 Kumar,Vyasa ended here
            bo.validateMatchingSSN(verifySsn);
            if (!bo.getMessages().hasErrorMessages()) {
                bo.save();
            } else {
            	Iterator iterator = bo.getMessages().getErrorMessages().iterator();
            	while(iterator.hasNext()) {
            		StringBuffer str = new StringBuffer("doUserProfileSave() failed for user profile id: "+entity.getId() + " ERROR:");
            		Message message = (Message)iterator.next();
        			str.append(message.getKey());
        			if (message.getInserts().size() > 0) {
        				str.append("; inserts=[").append(message.getInserts()).
        					append("]");
        			}
        			if (message.getProperties().size() > 0) {
        				str.append("; properties=[").append(message.getProperties()).
        					append("]");
        			}
        			if(LOG.isErrorEnabled()) {
        				LOG.error(str);
        			}
            	}
            }
        }
        return bo.getUserProfileValues();
    }

    private boolean verifyHintAnswers(UserProfile up, String hintAnswer1,
        String hintAnswer2)
    {
		//String expectedAnswer1 = StringUtils.lowerCase(up.getPasswordHintAnswer1());
    	String expectedAnswer1 =up.getPasswordHintAnswer1();
		//String expectedAnswer2 = StringUtils.lowerCase(up.getPasswordHintAnswer2());
    	String expectedAnswer2 =up.getPasswordHintAnswer2();
        boolean forgotUserNamePassword = false;
        if (StringUtils.isNotBlank(hintAnswer1) && hintAnswer1.endsWith("FP")) {
        	hintAnswer1 = StringUtils.substringBefore(hintAnswer1,"FP");
			forgotUserNamePassword = true;

        }
        //String actualAnswer1   = StringUtils.lowerCase(hintAnswer1);
		String actualAnswer1   =hintAnswer1;
		// String actualAnswer2   = StringUtils.lowerCase(hintAnswer2);
        String actualAnswer2   =hintAnswer2;

		if (forgotUserNamePassword)
			return (expectedAnswer1.equals(actualAnswer1) && expectedAnswer2
					.equals(actualAnswer2));

        return (expectedAnswer1.equals(actualAnswer1)
        		 || expectedAnswer2.equals(actualAnswer2));
            //only one question need to be answered correctly
    }

    /*
     * JAZZ:Task#20852 - Take out entry from eauthhash during deactivation for the user
     */
	public VoidServiceResponse deleteHashByUserProfileId(Long userProfileId) {
		VoidServiceResponse response = new VoidServiceResponse();
		EAuthHashServiceResponse hashRes = new EAuthHashServiceResponse ();
		Collection c = EAuthHashBaseBO.findEAuthHashByUserProfileId(userProfileId);
		if(c != null && !c.isEmpty()){
			Iterator it = c.iterator();
			while(it.hasNext()){
				EAuthHashBO e = (EAuthHashBO)it.next();
				hashRes = ServiceFactory.createEntityMaintenanceService().delete(e.getEAuthHashPK());
				Auditor.auditAccountDeactivateEauthHashEvent(userProfileId);
			}
		}
		if (hashRes.getMessages().hasErrorMessages()) {
			copyMessages(response, hashRes);
		}
		return response;
	}

	public gov.va.med.mhv.usermgmt.service.EAuthHashCollectionServiceResponse findEAuthByUserProfileId(Long userProfileId) {
		gov.va.med.mhv.usermgmt.service.EAuthHashCollectionServiceResponse response = new gov.va.med.mhv.usermgmt.service.EAuthHashCollectionServiceResponse();
        Collection<EAuthHashBO> bos = EAuthHashBO.findEAuthHashByUserProfileId(userProfileId);
        if (!bos.isEmpty()) {
	        Collection<gov.va.med.mhv.usermgmt.transfer.EAuthHash> tos = gov.va.med.mhv.usermgmt.bizobj.EAuthHashAssembler.getEAuthHashCollection(bos);
	        Iterator it = tos.iterator();
	        while(it.hasNext()) {
	        	gov.va.med.mhv.usermgmt.transfer.EAuthHash eauth = (gov.va.med.mhv.usermgmt.transfer.EAuthHash)it.next();
	        	response.addEAuthHash(eauth);
	        }
        }
        return response;
	}

	private String getVapiiWsdlUrl() {
		VapiiProperties properties = VapiiProperties.getInstance();
		String envType = properties.getVapiiEnvType();
		String wsdlUrl = null;
		ENVTYPES selectedEnv = null;
		 for (ENVTYPES envTypes: ENVTYPES.values()) {
			 if(envTypes.value().equalsIgnoreCase(envType))
				 selectedEnv = envTypes;
		 }
		//ENVTYPES envTypes	= ENVTYPES.(envType);
 		switch (selectedEnv) {
			case DEV:
				wsdlUrl = properties.getVapiiDevWsdlUrl();
				break;
			case INTB:
				wsdlUrl = properties.getVapiiIntbWsdlUrl();
				break;	
			case SYSTEST:	
				wsdlUrl = properties.getVapiiSysTestWsdlUrl();
				break;
			case PREPROD:	
				wsdlUrl = properties.getVapiiPreProdWsdlUrl();
				break;
			case PROD:	
				wsdlUrl = properties.getVapiiProdWsdlUrl();
				break;
			default:	
 		}
		if(LOG.isDebugEnabled()){
			LOG.debug("**************************  UserProfileServiceImpl.getVapiiWsdlUrl() wsdlUrl="+wsdlUrl);
		}

 		return wsdlUrl;

	}
	
	private UserIdType getVapiiUserIdType() {
		VapiiProperties properties = VapiiProperties.getInstance();
		if(LOG.isDebugEnabled()){
			LOG.debug("**************************  properties.getVapiiUserIdType()="+properties.getVapiiUserIdType());
		}
		if(properties.getVapiiUserIdType().equalsIgnoreCase(UserIdType.MHV.getValue())) {
			return UserIdType.MHV;
		} else if(properties.getVapiiUserIdType().equalsIgnoreCase(UserIdType.EDIPI.getValue())) {
			return UserIdType.EDIPI;
		} else if(properties.getVapiiUserIdType().equalsIgnoreCase(UserIdType.ICN.getValue())) {
			return UserIdType.ICN;
		}
		return null;
	}
	
	private String getVapiiUserId(UserProfile userProfile) {
		PatientService patientService = ServiceFactory.createPatientService();
		PatientServiceResponse patientForUserResponse = patientService.getPatientForUser(userProfile);
		Patient patient = patientForUserResponse.getPatient();
		
		VapiiProperties properties = VapiiProperties.getInstance();
		if(properties.getVapiiUserIdType().equalsIgnoreCase(UserIdType.MHV.getValue())) {
			if(LOG.isDebugEnabled()){
				LOG.debug("Returning UserProfileId="+userProfile.getId().toString());
			}
			return userProfile.getId().toString();
		} else if(properties.getVapiiUserIdType().equalsIgnoreCase(UserIdType.EDIPI.getValue())) {
			patientService = ServiceFactory.createPatientService();
			StringServiceResponse strResponse = 
				ServiceFactory.createMviIntegrationService().getEdipiFromMVI(patient);
			if(LOG.isDebugEnabled()){
				LOG.debug("Returning EDIPI="+strResponse.getString());
			}
			return strResponse.getString();
		} else if(properties.getVapiiUserIdType().equalsIgnoreCase(UserIdType.ICN.getValue())) {
			if(LOG.isDebugEnabled()){
				LOG.debug("Returning ICN="+patient.getIcn());
			}
			return patient.getIcn();
		}
		return null;
	}
	
	public StringServiceResponse getVapiiFormUrl() {
		VapiiProperties properties = VapiiProperties.getInstance();
		String envType = properties.getVapiiEnvType();
		String formUrl = null;
		ENVTYPES selectedEnv = null;
		 for (ENVTYPES envTypes: ENVTYPES.values()) {
			 if(envTypes.value().equalsIgnoreCase(envType))
				 selectedEnv = envTypes;
		 }
		//ENVTYPES envTypes	= ENVTYPES.(envType);
 		switch (selectedEnv) {
			case DEV:
				formUrl = properties.getVapiiDevFormUrl();
				break;
			case INTB:
				formUrl = properties.getVapiiIntbFormUrl();
				break;	
			case SYSTEST:	
				formUrl = properties.getVapiiSysTestFormUrl();
				break;
			case PREPROD:	
				formUrl = properties.getVapiiPreProdFormUrl();
				break;
			case PROD:	
				formUrl = properties.getVapiiProdFormUrl();
				break;
			default:	
 		}
		if(LOG.isDebugEnabled()){
			LOG.debug("**************************  UserProfileServiceImpl.getVapiiWsdlUrl() wsdlUrl="+formUrl);
		}
		StringServiceResponse stringResponse = new StringServiceResponse();
		stringResponse.setString(formUrl);
 		return stringResponse;
	}
	
	public StringServiceResponse getVapiiFormStatusForUser(
			UserProfile userProfile) {
		VapiiPartnerServicesProxy proxy = new VapiiPartnerServicesProxy(getVapiiWsdlUrl());//Use this one!!!
		GetFormStatusForUserRequest request = new GetFormStatusForUserRequest();
		User[] users = new User[1];
		users[0] = new User();
		
		users[0].setUserId(getVapiiUserId(userProfile));
		if(LOG.isDebugEnabled()){
			LOG.debug("**************************  UserProfileServiceImpl.getVapiiFormStatusForUser() getVapiiUserId(userProfile)="+getVapiiUserId(userProfile));
		}
		
		users[0].setUserIdType(getVapiiUserIdType());
		if(LOG.isDebugEnabled()){
			LOG.debug("**************************  UserProfileServiceImpl.getVapiiFormStatusForUser() getVapiiUserIdType()="+getVapiiUserIdType().getValue());
		}

		request.setUsers(users);
		StringServiceResponse stringResponse = new StringServiceResponse();
		stringResponse.setString("");
		try {
			GetFormStatusForUserResponse response = proxy.getFormStatusForUser(request);
			String responseStr = "";
			if(response.getFormReferences()!=null && response.getFormReferences()[0].getFormStatus() != null) {
				responseStr = response.getFormReferences()[0].getFormStatus().getValue();
			}
			stringResponse.setString(responseStr);
			if(LOG.isDebugEnabled()){
				LOG.debug("**************************  UserProfileServiceImpl.getVapiiFormStatusForUser() userProfile.getUserName():"+ userProfile.getUserName()+ "	formStatus:"+responseStr);
			}
			return stringResponse;
		} catch(Throwable t) {
			t.printStackTrace();
			addError(stringResponse, t.getMessage());
		}
		return stringResponse;
	}

	public BlobServiceResponse downloadVapiiFormForUser(
			UserProfile userProfile) {
		VapiiPartnerServicesProxy proxy = new VapiiPartnerServicesProxy(getVapiiWsdlUrl());//Use this one!!!
		GetDocumentForUserRequest request = new GetDocumentForUserRequest();
		GetFormStatusForUserRequest getFormStatusForUserRequest = new GetFormStatusForUserRequest();
		User[] users = new User[1];
		users[0] = new User();
		users[0].setUserId(getVapiiUserId(userProfile));
		users[0].setUserIdType(getVapiiUserIdType());

		getFormStatusForUserRequest.setUsers(users);
		StringServiceResponse stringResponse = new StringServiceResponse();
		stringResponse.setString("");
		try {
			GetFormStatusForUserResponse response = proxy.getFormStatusForUser(getFormStatusForUserRequest);
			if(response.getFormReferences()!=null && response.getFormReferences()[0] != null) {
				request.setDocumentId(response.getFormReferences()[0].getDocumentId());
				if(LOG.isDebugEnabled()){
					LOG.debug("response.getFormReferences()[0].getDocumentId():"+response.getFormReferences()[0].getDocumentId());
				}
			}
		} catch(Throwable t) {
			t.printStackTrace();
		}
		
		BlobServiceResponse blobResponse = new BlobServiceResponse();
		
		try {
			request.setUser(users[0]);
			request.setFormType(FormType.value1);
			GetDocumentForUserResponse response = proxy.getDocumentForUser(request);
			String responseStr = "";
			if(response.getDocument()!=null) {
				if(LOG.isDebugEnabled()){
					LOG.debug("response.getDocument()!=null");
				}
				blobResponse.setValue(response.getDocument());
			} else {
				if(LOG.isDebugEnabled()){
					LOG.debug("response.getDocument()==null");
				}
			}
		} catch(Throwable t) {
			t.printStackTrace();
		}
		return blobResponse;
	}
}